<?php
namespace MainBundle\ApiServices;

use MainBundle\ApiServices\Annotations\Validate;
use MainBundle\Data\ScoreSubmitType;
use MainBundle\Entity\Classwork;
use MainBundle\EntityManager\ClassroomManager;
use MainBundle\EntityManager\ClientManager;

class ClassroomService extends BaseService
{
    private $classroomManager;

    public function getClassroomManager(): ClassroomManager
    {
        if(empty($this->classroomManager)) {
            $this->classroomManager = new ClassroomManager($this->controller);
        }
        return $this->classroomManager;
    }

    protected function onAccessTokenSet()
	{
		$this->getClassroomManager()->initUser($this->getUser());
	}

    /**
	 * @Validate(register=true)
	 */
    public function serviceListCourses($args)
    {
        return ['list' => $this->getClassroomManager()->serviceListCourses()];
    }
    /**
	 * @Validate(register=true)
	 */
    public function serviceGetCourse($args)
    {
        return ['course' => $this->getClassroomManager()->serviceGetCourse($args['courseId'])];
    }

    /**
	 * @Validate(register=true)
	 */
    public function serviceListStudents($args)
    {
        return ['list' => $this->getClassroomManager()->serviceListStudents($args['courseId'])];
    }

    // database access ---------------------------------------------------
    /**
	 * @Validate(register=true)
	 */
    public function listMyClassrooms($args)
    {
        $list = $this->getClassroomManager()->listClassroomByOwner($this->getUser());
        return ['list' => $this->controller->listToJson($list)];
    }

    /**
	 * @Validate(register=true, secure=true)
	 */
    public function registerClassroom($args)
    {
        $classroom = $this->getClassroomManager()->createClassroom(
            $this->getUser(),
            $args['service'],
            $args['code'],
            $args['name'],
            $args['ojson']
        );
        
        return ['classroom' => $this->controller->objToJson($classroom)];
    }

    /**
	 * @Validate(register=true)
	 */
    public function getClassroom($args)
    {
        $classroom = $this->getClassroomManager()->getClassroom($args['id']);
        $this->designContract($classroom->getOwner()->getId() == $this->getUser()->getId(), 'Not your classroom');
        
        return ['classroom' => $this->controller->objToJson($classroom)];
    }

    /**
	 * @Validate(register=true)
	 */
    public function listClassroomStudents($args)
    {
        $classroom = $this->getClassroomManager()->getById($args['id']);
        $this->designContract($classroom->getOwner()->getId() == $this->getUser()->getId(), 'Not your classroom');
        $users = $this->getUserManager()->listByGoogleIds($args['googleIds']);
        
        return ['list' => $this->controller->listToJson($users)];
    }

    /**
     * @Validate(register=true, secure=true)
     */
    public function genRandomStudentScores($args)
    {
        $count = $args['count'];
        $clientCode = $args['client'];
        $classroomId = $args['classroomId'];
        $scoreKey = $args['scoreKey'];
        $scoreService = new ScoreService($this->controller);
        $clientManager = new ClientManager($this->controller);
        $client = $clientManager->getByCode($clientCode);
        $users = $this->getClassroomManager()->listClassroomStudents($classroomId);
        while($count-- > 0) {
            $user = $users[array_rand($users)];
            $scoreService->setAccessToken($client, $user);
            $days = rand(1, 30);
            $scoreService->_submitTime = new \DateTime("-$days days");
            $score = rand(1, 100);
            $scoreService->submitScore([
                'scoreKey' => $scoreKey,
                'score' => $score,
                'submitType' => ScoreSubmitType::SUBMIT_ADD_SINCE,
            ]);
        }
        return ['done' => true];
    }

    /**
     * @Validate(register=true)
     */
    public function listLicensedClients($args)
    {
        $this->getClassroomManager()->initUser($this->user);
        $wpUserId = $this->getClassroomManager()->getWpUserId();
        $clientCodes = $this->getClassroomManager()->getLicenseManager()->listLicensedContentCodes($wpUserId);
        $clients = $this->getClientManager()->listByCodes($clientCodes);
        return ['list' => $this->controller->listToJson($clients)];
    }

    /**
     * @Validate(register=true, secure=true)
     */
    public function listLicensedClasswork($args)
    {
        $classroomIds = $args['classroomIds'];
        $clientCodes = $args['clientCodes'];
        $classworkList = [];
        foreach($classroomIds as $classroomId) {
            $thisList = $this->getClassroomManager()->listClassworkByClassroomId($classroomId);
            $registeredClientCodes = [];
            /** @var Classwork $classwork */
            foreach($thisList as $classwork) {
                $registeredClientCodes[] = $classwork->getClient()->getCode();
            }

            foreach($clientCodes as $clientCode) {
                if(!in_array($clientCode, $registeredClientCodes)) {
                    $classwork = $this->getClassroomManager()->createClasswork($classroomId, $clientCode);
                    $thisList[] = $classwork;
                }
            }
            
            $classworkList = array_merge($classworkList, $thisList);
        }

        return ['list' => $this->controller->listToJson($classworkList)];
    }
    /**
     * @Validate(register=true)
     */
    public function listStudentClasswork($args)
    {
        $classworkList = $this->getClassroomManager()->listClassworkByStudent();
        return ['list' => $this->controller->listToJson($classworkList)];
    }

    /**
     * @Validate(register=true, secure=true)
     */
    public function turninStudentClasswork($args)
    {
        $classwork = $this->getClassroomManager()->getStudentClasswork($this->getClient()->getCode());
        $message = $args['message'];
        $response = $this->getClassroomManager()->turnInClasswork($classwork, $message);
        return ['response' => 'done'];
    }
    
}