<?php

namespace MainBundle\ApiServices;

use MainBundle\Controller\BaseApiController;
use MainBundle\ApiServices\Annotations\Validate;
use MainBundle\Data\OrderType;
use MainBundle\EntityManager\UserScoreHistoryManager;
use MainBundle\EntityManager\UserScoreMonthlyManager;
use MainBundle\EntityManager\UserScoreWeeklyManager;
use MainBundle\EntityManager\UserScoreDailyManager;
use MainBundle\EntityManager\AbstractUserScoreManager;
use MainBundle\Entity\Abstracts\AbstractUserScore;
use MainBundle\Entity\Client;
use MainBundle\Entity\User;
use MainBundle\Data\TimeRange;

class ScoreService extends BaseService
{
	private $userScoreManagers;

	/**
	 * for dev only, when random submit scores for testing
	 */
	public $_submitTime = null;

	public function __construct(BaseApiController $controller)
	{
		parent::__construct($controller);
		$this->userScoreManagers = array(
			TimeRange::HISTORY => new UserScoreHistoryManager($controller),
			TimeRange::MONTHLY => new UserScoreMonthlyManager($controller),
			TimeRange::WEEKLY => new UserScoreWeeklyManager($controller),
			TimeRange::DAILY => new UserScoreDailyManager($controller)
		);
	}

	public function getDefaultScoreManager(): AbstractUserScoreManager
	{
		return $this->userScoreManagers[TimeRange::HISTORY];
	}

	public function getUserScoreManager($timeRange): AbstractUserScoreManager
	{
		return $this->userScoreManagers[$timeRange];
	}

	private function _submitScore(Client $client, User $user, $scoreKey, $score, $submitType)
	{
		$scoreMode = $this->getDefaultScoreManager()->getScoreMode($client, $scoreKey, true);
		$this->designContract(!empty($scoreMode), 'The scoreKey does not exist.');
		$this->designContract($scoreMode->getToDollarRate() == 0, 'This scoreKey is of a game currency, use "submitGameDollar" instead.');

		$scoreMap = array('scoreKey' => $scoreMode->getScoreKey());
		$this->designContract(!$scoreMode->getDisabled(), 'The score key is disabled.');

		/** @var AbstractUserScoreManager $usManager */
		foreach ($this->userScoreManagers as $timeRange => $usManager) {
			if(!empty($this->_submitTime)) {
				$usManager->_debugNow = $this->_submitTime;
			}
			$result = $usManager->submitScore($user, $scoreMode, $score, $submitType, false);
			/** @var AbstractUserScore $userScore */
			$userScore = $result['userScore'];
			$scoreMap[$timeRange] = $this->controller->objToJson($userScore);
		}
		return $scoreMap;
	}


	/**
	 * @Validate(secure=true, register=true)
	 */
	public function submitScore($args)
	{
		$scoreKey = $args['scoreKey'];
		$score = $args['score'];
		$submitType = $args['submitType'];

		$user = $this->getUser();
		$client = $this->getClient();
		$result = $this->_submitScore($client, $user, $scoreKey, $score, $submitType);
		$this->getDefaultScoreManager()->flush();
		return $result;
	}

	/**
	 * @Validate(register=true)
	 */
	public function getScore($args)
	{
		$scoreKey = $args['scoreKey'];
		$timeRange = $args['timeRange'];
		$timestamp = $args['timestamp'];

		$time = new \DateTime();
		if ($timestamp > 0) {
			$time->setTimestamp($timestamp);
		}

		$user = $this->getUser();
		$client = $this->getClient();
		$usManager = $this->getUserScoreManager($timeRange);
		$scoreMode = $usManager->getScoreMode($client, $scoreKey, false);
		if (empty($scoreMode)) {
			$userScore = null;
		} else {
			$this->designContract($scoreMode->notDeleted(), 'The score key is deleted.');
			$userScore = $usManager->getUserScore($user, $scoreMode, $time);
		}
		return ['userScore' => $userScore ? $userScore->exportJson() : null];
	}

	public function listScores($args)
	{
		$scoreKey = $args['scoreKey'];
		$timeRange = $args['timeRange'];
		$orderType = $args['orderType'];
		$timestamp = $args['timestamp'];
		$start = $args['start'];
		$length = $args['length'];

		$client = $this->getClient();
		$time = new \DateTime();
		if ($timestamp > 0) {
			$time->setTimestamp($timestamp);
		} else if($orderType == OrderType::NEW_TO_OLD) {
			$time = null;
		}
		$usManager = $this->getUserScoreManager($timeRange);
		$scoreMode = $usManager->getScoreMode($client, $scoreKey, false);
		$list = array();
		if (!empty($scoreMode)) {
			$this->designContract($scoreMode->notDeleted(), 'The score key is deleted.');
			$scores = $usManager->listScores($scoreMode, $orderType, $start, $length, $time);
			$list = $this->controller->listToJson($scores, array('basic','full'));
		}

		/** @var AbstractUserScore $score */
		return $usManager->generateScoreListResult($list, 1, empty($time) ? null : $usManager->getPeriodStartByTime($time));
	}

	public function getScoresTotal($args)
	{
		$scoreKey = $args['scoreKey'];
		$timeRange = $args['timeRange'];
		$timestamp = $args['timestamp'];

		$client = $this->getClient();
		$time = new \DateTime();
		if ($timestamp > 0) {
			$time->setTimestamp($timestamp);
		}
		$usManager = $this->getUserScoreManager($timeRange);
		$scoreMode = $usManager->getScoreMode($client, $scoreKey, false);
		if (empty($scoreMode)) {
			$total = 0;
		} else {
			$this->designContract($scoreMode->notDeleted(), 'The score key is deleted.');
			$total = $usManager->getScoresTotal($scoreMode, $time);
		}
		return ['total' => $total];
	}

	/**
	 * @Validate(register=true)
	 */
	public function getScoreAndRank($args)
	{
		$scoreKey = $args['scoreKey'];
		$timeRange = $args['timeRange'];
		$orderType = $args['orderType'];
		$timestamp = $args['timestamp'];

		$user = $this->getUser();
		$client = $this->getClient();

		$time = new \DateTime();
		if ($timestamp > 0) {
			$time->setTimestamp($timestamp);
		}
		$usManager = $this->getUserScoreManager($timeRange);
		$scoreMode = $usManager->getScoreMode($client, $scoreKey, false);
		if (empty($scoreMode)) {
			$userScore = null;
		} else {
			$this->designContract($scoreMode->notDeleted(), 'The score key is deleted.');
			$userScore = $usManager->getUserScore($user, $scoreMode, $time);
		}
		if (empty($userScore)) {
			$json = array(
				'score' => null,
				'rank' => -1
			);
		} else {
			$rank = $usManager->getScoreRank($scoreMode, $userScore->getScore(), $orderType, $time);
			$json = array(
				'score' => $userScore->exportJson(),
				'rank' => $rank,
				'periodStart' => $usManager->getPeriodStartByTime($time)->getTimestamp()
			);
		}
		return $json;
	}

	/**
	 * @Validate(register=true)
	 */
	public function listScoresAndRankAroundUser($args)
	{
		$scoreKey = $args['scoreKey'];
		$timeRange = $args['timeRange'];
		$orderType = $args['orderType'];
		$timestamp = $args['timestamp'];
		$shiftStart = $args['shiftStart'];
		$length = $args['length'];

		$user = $this->getUser();
		$client = $this->getClient();

		$time = new \DateTime();
		if ($timestamp > 0) {
			$time->setTimestamp($timestamp);
		}

		$usManager = $this->getUserScoreManager($timeRange);
		$scoreMode = $usManager->getScoreMode($client, $scoreKey, false);
		if (empty($scoreMode)) {
			$result = $usManager->generateScoreListResult(array(), 1, $usManager->getPeriodStartByTime($time));
		} else {
			$this->designContract($scoreMode->notDeleted(), 'The score key is deleted.');
			$result = $usManager->listScoresAndRankAroundUser($scoreMode, $user, $orderType, $time, $shiftStart, $length);
		}
		return $result;
	}

	public function listScoresByUsernames($args)
	{
		$scoreKey = $args['scoreKey'];
		$usernames = $args['usernames'];
		$timeRange = $args['timeRange'];
		$timestamp = $args['timestamp'];

		$client = $this->getClient();
		$time = new \DateTime();
		if ($timestamp > 0) {
			$time->setTimestamp($timestamp);
		}
		$usManager = $this->getUserScoreManager($timeRange);
		$scoreMode = $usManager->getScoreMode($client, $scoreKey, false);
		$list = array();
		if (!empty($scoreMode)) {
			$this->designContract($scoreMode->notDeleted(), 'The score key is deleted.');
			if (!empty($usernames)) {
				$scores = $usManager->listScoresByUsernames($scoreMode, $usernames, $time);
				$list = $this->controller->listToJson($scores);
			}
		}

		return array(
			'list' => $list
		);
	}

	/**
	 * @Validate(register=true)
	 */
	public function listScoresHistory($args)
	{
		$scoreKey = $args['scoreKey'];
		$timeRange = $args['timeRange'];
		$startTimestamp = $args['startTimestamp'];
		$endTimestamp = $args['endTimestamp'];

		$user = $this->getUser();
		$client = $this->getClient();
		$startTime = new \DateTime();
		if (!empty($startTimestamp) && $startTimestamp > 0) {
			$startTime->setTimestamp($startTimestamp);
		} else {
			$startTime->modify('-1 month');
		}
		$endTime = new \DateTime();
		if (!empty($endTimestamp) && $endTimestamp > 0) {
			$endTime->setTimestamp($endTimestamp);
		}
		$usManager = $this->getUserScoreManager($timeRange);
		$scoreMode = $usManager->getScoreMode($client, $scoreKey, false);
		$list = [];
		if (!empty($scoreMode)) {
			$this->designContract($scoreMode->notDeleted(), 'The score key is deleted.');
			$scores = $usManager->listScoresHistory($user, $scoreMode, $startTime, $endTime);
			$list = $this->controller->listToJson($scores, array('basic','full'));
		}
		return array(
			'list' => $list
		);
	}

	/**
	 * @Validate(register=true)
	 */
	public function listScoresHistoryByUsernames($args)
	{
		$clientCode = $args['clientCode']; 
		$usernames = $args['usernames'];
		$scoreKey = $args['scoreKey'];
		$timeRange = $args['timeRange'];
		$startTimestamp = $args['startTimestamp'];
		$endTimestamp = $args['endTimestamp'];

		$client = $this->getClientManager()->getByCode($clientCode);
		$startTime = new \DateTime();
		if (!empty($startTimestamp) && $startTimestamp > 0) {
			$startTime->setTimestamp($startTimestamp);
		} else {
			$startTime->modify('-1 month');
		}
		$endTime = new \DateTime();
		if (!empty($endTimestamp) && $endTimestamp > 0) {
			$endTime->setTimestamp($endTimestamp);
		}
		$usManager = $this->getUserScoreManager($timeRange);
		$scoreMode = $usManager->getScoreMode($client, $scoreKey, false);
		$list = [];
		if (!empty($scoreMode)) {
			$this->designContract($scoreMode->notDeleted(), 'The score key is deleted.');
			$scores = $usManager->listScoresHistoryByUsernames($usernames, $scoreMode, $startTime, $endTime);
			$list = $this->controller->listToJson($scores, array('basic','full'));
		}
		return array(
			'list' => $list
		);
	}

}
