<?php

namespace MainBundle\Controller;

use MainBundle\EntityManager\ClientManager;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\HttpFoundation\Request;

use Doctrine\Common\Annotations\AnnotationReader;
use Doctrine\Common\Annotations\FileCacheReader;

use MainBundle\EntityManager\UserManager;
use MainBundle\ApiServices\Annotations\Validate;
use MainBundle\ApiServices\ScoreService;
use MainBundle\Data\ScoreSubmitType;
use MainBundle\EntityManager\ClassroomManager;

class ApiController extends BaseApiController
{
    /**
     * @Route("/get/app_config/{appName}", name="get_app_config")
     */
    public function getAppConfigAction(Request $request, $appName)
    {
        $systemManager = new \MainBundle\EntityManager\SystemManager($this);
        $AmBox = $systemManager->generateIdeAmBox($appName, true);
        return $this->createJsonRespons(array(
            'AmBox' => base64_encode(json_encode($AmBox)),
        ));
    }

    /**
     * @Route("/test")
     */
    public function testAction(Request $request)
    {
        return $this->createJsonRespons(array(
            'value' => $this->getYmlParameter('google_app_id')
        ));
    }

    
    /**
     * @Route("/gen/scores/guest/random/{count}")
     */
    public function genGuestRandomScoresAction($count)
    {
        $count = intval($count);
        $scoreService = new ScoreService($this);
        $userManager = new UserManager($this);
        $clientManager = new ClientManager($this);
        $client = $clientManager->getByCode('co2-calculator');
        $users = [];
        while(count($users) < $count) {
            $user = $userManager->getAnonymousRandomUser();
            $scoreService->setAccessToken($client, $user);
            $score = rand(500, 5000) / 100;
            $scoreService->submitScore('co2saving', $score, ScoreSubmitType::SUBMIT_ADD_SINCE);
            $users[] = $user;
        }
        return $this->generateJsonResponse(array(
            'result' => 1
        ));
    }

    /**
     * @Route("/get/wp_login_user")
     */
    public function getWpUserAction()
    {
        $this->getWpServer()->loadWordpress();
        $user = \wp_get_current_user();
        $userManager = new UserManager($this);
        return $this->generateJsonResponse(array(
            'user' => empty($user) ? null : $this->objToJson($userManager->convertWpUserToUser($user))
        ));
    }

    /**
     * @Route("/login")
     */
    public function loginClientAction(Request $request)
    {
        $post = json_decode(urldecode($request->getContent()), true);
        $username = $post['username'];
        $userManager = new UserManager($this);
        $user = $userManager->getByUsername($username);
        return $this->generateJsonResponse(array(
            'token' => $user->getAccessToken(),
            'user' => $this->objToJson($user)
        ));
    }

    /**
     * @Route("/validate_token")
     */
    public function validateUserToken(Request $request)
    {
        $post = json_decode(urldecode($request->getContent()), true);
        $userToken = $post['userToken'];
        $userManager = new UserManager($this);
        $user = $userManager->validateUserToken($userToken);
        return $this->generateJsonResponse(array(
            'user' => $this->objToJson($user)
        ));
    }
    /**
     * @Route("/get/page/{slug}")
     */
    public function getPageContent($slug)
    {
        $this->getWpServer()->loadWordpress();
        $page = \get_page_by_title($slug);
        $this->designContract(!empty($page), 'page not found');
        return $this->generateJsonResponse(array(
            'content' => $page->post_content
        ));
    }



    /**
     * @Route("/service")
     */
    public function serviceAction(Request $request)
    {
        $post = json_decode(urldecode($request->getContent()), true);
        $now = time();
        $command = $this->parseCommand($post, $now);
        $result = $this->executeCommand($command, $command['secure']);
        return $this->generateJsonResponse(array(
            'result' => $result,
			'timestamp' => $now,
            'error' => '',
        ));
    }

    /**
     * @Route("/service/test/score")
     */
    public function serviceTestScoreAction(Request $request)
    {
        $now = time();
        $userManager = new UserManager($this);
        $user = $userManager->getById(1);
        $userToken = $user->getAccessToken();
        $client = 'aaa';
        $service = 'score';
        $method = 'submitScore';
        $args = json_encode(['scoreKey' => 'exp', 'score' => 1.63, 'submitType' => ScoreSubmitType::SUBMIT_ADD]);
        $timeCode = base_convert("" . $now, 10, 16);
        $uid = uniqid();
        $sigKey = $this->generateCommandSignatureKey($client, $service, $method, $args, $uid, $timeCode, $userToken);
        $sig = md5($sigKey);

        $command = $this->parseCommand(array(
            'userToken' => $userToken,
            'client' => $client,
            'service' => $service,
            'method' => $method,
            'timeCode' => $timeCode,
            'args' => $args,
            'uid' => $uid,
            'sig' => $sig,
        ), $now);
        $result = $this->executeCommand($command, true);
        return $this->generateJsonResponse(array(
            'result' => $result,
            'timestamp' => $now,
            'error' => '',
        ));
    }

    /**
     * @Route("/service/test/state")
     */
    public function serviceTestStateAction(Request $request)
    {
        $now = time();
        $userManager = new UserManager($this);
        $user = $userManager->getById(1);
        $userToken = $user->getAccessToken();
        $client = 'aaa';
        $service = 'state';
        $method = 'setState';
        $args = json_encode(['category' => 'testCat', 'key' => 'me', 'value' => ['atk'=> 11, 'def'=>8,'name'=>'cool']]);
        $timeCode = base_convert("" . $now, 10, 16);
        $uid = uniqid();
        $sigKey = $this->generateCommandSignatureKey($client, $service, $method, $args, $uid, $timeCode, $userToken);
        $sig = md5($sigKey);

        $command = $this->parseCommand(array(
            'userToken' => $userToken,
            'client' => $client,
            'service' => $service,
            'method' => $method,
            'timeCode' => $timeCode,
            'args' => $args,
            'uid' => $uid,
            'sig' => $sig,
        ), $now);
        $result = $this->executeCommand($command, true);
        return $this->generateJsonResponse(array(
            'result' => $result,
            'timestamp' => $now,
            'error' => '',
        ));
    }

    private function parseCommand($command, $now)
    {
        $cacheRoot = $this->getService('kernel')->getCacheDir();
        $annotationReader = new FileCacheReader(new AnnotationReader(), $cacheRoot . '/gltapi_annotation', $this->isDev());
        $cmdHistoryDir = $cacheRoot . '/cmd_history';

        $token = $command['userToken'];
        $clientCode = $command['client'];
        $serviceName = $command['service'];
        $method = $command['method'];
        $timeCode = $command['timeCode'];
        $args = $command['args'];

        $userManager = new UserManager($this);
        $user = null;
        if (!empty($token)) {
            $user = $userManager->validateUserToken($token);
        }

        $clientManager = new ClientManager($this);
        $client = $clientManager->getOrCreateByCode($clientCode);

        $time = base_convert($timeCode, 16, 10);

        $this->designContract($time <= $now, "This command came from future: $serviceName.$method " . ($time - $now));

        $serviceClass = 'MainBundle\\ApiServices\\' . ucfirst($serviceName) . 'Service';
        $this->designContract(class_exists($serviceClass), "The service does not exist: $serviceName");

        try {
            $annotations = $annotationReader->getMethodAnnotations(new \ReflectionMethod($serviceClass,  $method));
        } catch (\ReflectionException $e) {
            $this->designContract(false, "The method does not exist: $serviceName.$method");
        }

        /** @var \MainBundle\ApiServices\BaseService $service */
        $service = new $serviceClass($this);
        $validate = new Validate(array());

        foreach ($annotations as $ann) {
            if ($ann instanceof Validate) {
                $validate = $ann;
            }
        }

        if ($validate->time) {
            // client needs to update server time every 5 minutes
            // if the time was 10 minutes(600 secs) ago, this is very likely an attack or resend by hacker
            $this->designContract($time + 600 >= $now, "This command was expired: $serviceName.$method");
        }

        if (($validate->register || $validate->admin) && empty($user)) {
            $this->designContract(false, "Guest prohibited: $serviceName.$method");
        }

        if ($validate->admin && !$user->isSuperAdmin()) {
            $this->designContract(false, "Admin only command: $serviceName.$method");
        }

        $service->setAccessToken($client, $user);

        if ($validate->secure) {
            $uid = empty($command['uid']) ? '' : $command['uid'];
            $sig = empty($command['sig']) ? '' : $command['sig'];
            $sigKey = $this->generateCommandSignatureKey($client->getCode(), $serviceName, $method, $args, $uid, $timeCode, $token);

            $this->designContract(md5($sigKey) == $sig, "This command was modified illigally");

            $historyFilename = $cmdHistoryDir . '/' . substr($uid, 0, 3) . '.txt';
            if (file_exists($historyFilename)) {
                if (strpos(file_get_contents($historyFilename), $sig) !== false) {
                    $this->designContract(false, "This command had been called: $serviceName.$method");
                }
                // 2 MB max
                file_put_contents($historyFilename, "$sig\n", filesize($historyFilename) > 2000000 ? 0 : FILE_APPEND);
            } else {
                @mkdir($cmdHistoryDir, 0777, true);
                file_put_contents($historyFilename, "$sig\n");
            }
        }
        return array(
            'serviceClass' => $service,
            'secure' => $validate->secure,
            'method' => $method,
            'args' => json_decode($args, true),
            'cmd' => $serviceName . '.' . $method,
        );
    }

    private function generateCommandSignatureKey($client, $serviceName, $method, $args, $uid, $timeCode, $token)
    {
        $key = $client . '|' . $serviceName . '.' . $method . $args . $uid . '|bay|' . $timeCode . '/' . $token . '{spring}.1';
        return str_replace(["'", '"'], ['',''], $key);
    }

    private function executeCommand($command, $transaction)
    {
        if ($transaction) {
            $this->getDocMgr()->getConnection()->beginTransaction();
        }

        $cmd = array();

        try {
            $cmd = $command;
            $result = call_user_func_array(array($command['serviceClass'], $command['method']), [$command['args']]);

            if ($transaction) {
                $this->getDocMgr()->getConnection()->commit();
            }
            return $result;
        } catch (\Throwable $e) {
            $this->onServiceError($cmd, $transaction, $e);
        }
    }

    private function onServiceError($command, $transaction, $exception)
    {
        if ($transaction) {
            $this->getDocMgr()->getConnection()->rollBack();
        }
        $this->designContract(false, $exception->getMessage());
    }
}
