<?php

namespace MainBundle\Controller;

use Doctrine\Common\Annotations\AnnotationReader;
use MainBundle\Data\LocaleInfo;
use MainBundle\Services\WpService;
use MainBundle\Utils\StringUtil;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Serializer\Mapping\Factory\ClassMetadataFactory;
use Symfony\Component\Serializer\Mapping\Loader\AnnotationLoader;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;

class ControllerBase extends Controller implements IControllerBase
{

    private $serializer;

    /**
	 * @return string
	 */
	public function getRootDir()
	{
		return $this->getService('kernel')->getRootDir();
    }
    
    public function generateUrlWithUri($uri)
    {
        $refUrl = $this->generateUrl('phpinfo', ['password'=>'anything']);
        $url = StringUtil::removeEnd($refUrl, '/phpinfo/anything');
        return $url .= $uri;
    }

    private function getSerializer()
    {
        if (empty($this->serializer)) {
            $convertDateTime = function ($dateTime) {
                return $dateTime instanceof \DateTime
                    ? $dateTime->getTimestamp()
                    : '';
            };

            $compressedText = function($text) {
                if(!empty($text) && strpos($text, '/*gzip*/') === 0) {
                    $text = base64_decode(substr($text, strlen('/*gzip*/')));
                    $text = gzuncompress($text);
                }
                return $text;
            };

            $filterNotes = function($notes) {
                if(is_string($notes)) {
                    return $notes;
                }
                $list = array();
                for($i=0;$i<$notes->count();++$i) {
                    $note = $notes->get($i);
                    if(!$note->getDisabled()) {
                        $list[] = $this->objToJson($note, array('basic'));
                    }
                }
                return $list;
            };

            $timeCallback = array(
                'time' => $convertDateTime,
                'createTime' => $convertDateTime,
                'submitTime' => $convertDateTime,
                'editTime' => $convertDateTime,
                'completeTime' => $convertDateTime,
                'notes' => $filterNotes,

                'socialMediaLinks' => $compressedText,
                'biography' => $compressedText,
                'professionalSkills' => $compressedText,
                'filmmakerInspireStory' => $compressedText,
                'directorInspireStory' => $compressedText,
                'filmInDev' => $compressedText,
                'outlineIdealCareer' => $compressedText,
                'professionalReferences' => $compressedText
            );

            $classMetadataFactory = new ClassMetadataFactory(new AnnotationLoader(new AnnotationReader()));
            $normalizer = new ObjectNormalizer($classMetadataFactory);
            $normalizer->setCallbacks($timeCallback);
            $this->serializer = new Serializer(array($normalizer), array(new JsonEncoder()));
        }
        return $this->serializer;
    }

    public function objToJson($obj, $groups = null)
    {
        if($obj === null) {
            return null;
        }
        if(empty($groups)) {
            $groups = array('basic', 'full');
        }
        return $this->getSerializer()->normalize($obj, null, array('groups' => $groups));
    }
    public function listToJson($list, $groups = null)
    {
        if(empty($groups)) {
            $groups = array('basic');
        }

        $json = array();
        foreach ($list as $obj) {
            $json[] = $this->objToJson($obj, $groups);
        }
        return $json;
    }

    protected function generateJsonResponse($json)
    {
        $response = new JsonResponse($json);
        $response->setEncodingOptions($response->getEncodingOptions() | JSON_PRETTY_PRINT);
        return $response;
    }

    public function getService($service)
    {
        if ($service == 'aws.s3') {
            return $this->get('aws_service')->getS3Client();
        }
        return $this->get($service);
    }

    public function getWpServer(): WpService {
        return $this->getService('wp_service');
    }

    /**
     * @return Controller
     */
    public function getThis()
    {
        return $this;
    }

    public function isDev()
    {
        return $this->container->getParameter('kernel.environment') == 'dev';
    }

    /**
     *
     * {@inheritDoc}
     * @see \MainBundle\Controller\IControllerBase::getMyUser()
     */
    public function getMyUser()
    {
        if ($this->isDev()) {
            /*
        $userManager = new UserManager($this);
        $user = $userManager->getByEmail('haskasu@gmail.com');
        return $user;
         */
        }
        return $this->getUser();
    }

    public function getYmlParameter($key)
    {
        return $this->getParameter($key);
    }

    public function getObject($key)
    {
        return $this->get($key);
    }

    public function purifyHTML($string)
    {
        $purifier = $this->container->get('html_purifier_service')->purifier;
        return $purifier->purify($string);
    }

    public function getLocale(Request $request)
    {
        if ($request->query->has('locale')) {
            $locale = $this->convertLocale($request->query->get('locale'));
            return LocaleInfo::getByCode($locale);
        }
        if ($request->cookies->has('userLocale')) {
            $locale = $this->convertLocale($request->cookies->get('userLocale'));
            return LocaleInfo::getByCode($locale);
        }
        return LocaleInfo::getDefault();
    }

    private function convertLocale($locale)
    {
        if (StringUtil::startsWith($locale, 'fr')) {
            return 'fr';
        }
        return 'en';
    }

}
