<?php

namespace MainBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use FOS\UserBundle\Model\User as BaseUser;
use MainBundle\Utils\StringUtil;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * User
 *
 * @ORM\Table
 * @ORM\Entity
 */
class User extends BaseUser
{
    /**
     * @var integer
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    /**
     * google / facebook / etc...
     * @var string
     * 
     * @ORM\Column(type="string", length=31)
     */
    protected $service;

    /** @ORM\Column(name="google_id", type="string", length=255, nullable=true) */
    protected $google_id;

    /** @ORM\Column(name="google_access_token", type="string", length=255, nullable=true) */
    protected $google_access_token;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=127)
     */
    private $nickname;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=127, nullable=true)
     */
    private $iconFile;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=255, nullable=true)
     */
    private $profileUrl;

    /**
     * @var \Date $birthday
     *
     * @ORM\Column(name="birthday", type="date", nullable=true)
     */
    private $birthday;

    /**
     * @var number $settings
     *
     * @ORM\Column(name="settings", type="integer")
     */
    private $settings = 0;

    /**
     * @var boolean $blocked
     *
     * @ORM\Column(name="blocked", type="boolean")
     */
    private $blocked = false;

    /**
     * @ORM\Column(type="string", length=2047)
     */
    private $metaJson = '{}';

    /**
     * @ORM\OneToMany(targetEntity=Classroom::class, mappedBy="owner")
     */
    private $classrooms;

    /**
     * @Groups({"basic"})
     */
    public function getAdmin()
    {
        return $this->isSuperAdmin() ? 1 : 0;
    }
    /**
     * @Groups({"basic"})
     */
    public function getIcon()
    {
        return $this->getIconUri();
    }

    /**
     * @Groups({"basic"})
     */
    public function getUsername()
    {
        return parent::getUsername();
    }

    /**
     * @Groups({"basic"})
     */
    public function getEmail()
    {
        return parent::getEmail();
    }

    public function getAccessToken()
    {
        $gene = $this->password;
        return $this->username . '|' . md5($this->id . '|' . $this->username . '|' . $gene . '|4lksd90*^3a');
    }

    public function getIconUri()
    {
        if (empty($this->iconFile))
            return '/img/default_usericon.jpg';
        if (strpos($this->iconFile, 'http') === 0)
            return $this->iconFile;
        return '/img/user/u' . $this->getId() . '/' . $this->getIconFile();
    }

    /**
     * Get id
     *
     * @return integer 
     */
    public function getId()
    {
        return $this->id;
    }

    public function isGuest()
    {
        return StringUtil::endsWith($this->username, '@guest', false);
    }

    /**
     * Set nickname
     *
     * @param string $nickname
     * @return User
     */
    public function setNickname($nickname)
    {
        $this->nickname = $nickname;

        return $this;
    }
    /**
     * Get nickname
     *
     * @Groups({"basic"})
     * @return string
     */
    public function getNickname()
    {
        return $this->nickname;
    }

    /**
     * Set iconFile
     *
     * @param string $iconFile
     * @return User
     */
    public function setIconFile($iconFile)
    {
        $this->iconFile = $iconFile;

        return $this;
    }
    /**
     * Get iconFile
     *
     * @return string
     */
    public function getIconFile()
    {
        return $this->iconFile;
    }

    /**
     * Set google_id
     *
     * @param string $googleId
     * @return User
     */
    public function setGoogleId($googleId)
    {
        $this->google_id = $googleId;

        return $this;
    }

    /**
     * Get google_id
     *
     * @return string 
     */
    public function getGoogleId()
    {
        return $this->google_id;
    }

    /**
     * Set google_access_token
     *
     * @param string $googleAccessToken
     * @return User
     */
    public function setGoogleAccessToken($googleAccessToken)
    {
        $this->google_access_token = $googleAccessToken;

        return $this;
    }

    /**
     * Get google_access_token
     *
     * @return string 
     */
    public function getGoogleAccessToken()
    {
        return $this->google_access_token;
    }

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        $this->votes = new ArrayCollection();
        $this->classrooms = new ArrayCollection();
    }


    /**
     * Set service
     *
     * @param string $service
     *
     * @return User
     */
    public function setService($service)
    {
        $this->service = $service;

        return $this;
    }

    /**
     * Get service
     *
     * @return string
     */
    public function getService()
    {
        return $this->service;
    }

    /**
     * Set birthday
     *
     * @param \DateTime $birthday
     *
     * @return User
     */
    public function setBirthday($birthday)
    {
        $this->birthday = $birthday;

        return $this;
    }

    /**
     * Get birthday
     *
     * @return \DateTime
     */
    public function getBirthday()
    {
        return $this->birthday;
    }

    /**
     * Set settings
     *
     * @param integer $settings
     *
     * @return User
     */
    public function setSettings($settings)
    {
        $this->settings = $settings;

        return $this;
    }

    /**
     * Get settings
     *
     * @Groups({"basic"})
     * @return integer
     */
    public function getSettings()
    {
        return $this->settings;
    }

    public function hasSetting($setting)
    {
        return !empty($this->settings & $setting);
    }

    /**
     * Set profileUrl
     *
     * @param string $profileUrl
     *
     * @return User
     */
    public function setProfileUrl($profileUrl)
    {
        $this->profileUrl = $profileUrl;

        return $this;
    }

    /**
     * Get profileUrl
     *
     * @return string
     */
    public function getProfileUrl()
    {
        return $this->profileUrl;
    }

    /**
     * Set blocked
     *
     * @param boolean $blocked
     *
     * @return User
     */
    public function setBlocked($blocked)
    {
        $this->blocked = $blocked;

        return $this;
    }

    /**
     * Get blocked
     *
     * @Groups({"basic"})
     * @return boolean
     */
    public function getBlocked()
    {
        return $this->blocked;
    }

    
    /**
     * Set metaJson
     *
     * @param string $metaJson
     */
    public function setMetaJson($metaJson): self
    {
        if (strlen($metaJson) > 1000) {
            $compressedMetaJson = '/*gzip*/' . base64_encode(gzcompress($metaJson));
            if (strlen($compressedMetaJson) > 1000) {
                $meta = json_decode($metaJson);
                $safeMeta = array();
                $metaJson = json_encode($safeMeta);
            } else {
                $metaJson = $compressedMetaJson;
            }
        }
        $this->metaJson = $metaJson;

        return $this;
    }

    /**
     * Get metaJson
     *
     * @return string
     */
    public function getMetaJson()
    {
        if (!empty($this->metaJson) && strpos($this->metaJson, '/*gzip*/') === 0) {
            $metaJson = base64_decode(substr($this->metaJson, strlen('/*gzip*/')));
            $metaJson = gzuncompress($metaJson);
            return $metaJson;
        }
        return $this->metaJson;
    }

    public function setMeta($meta): self
    {
        if (empty($meta)) {
            $meta = array();
        }
        $this->setMetaJson(json_encode($meta));
        return $this;
    }

    public function getMeta()
    {
        $json = $this->getMetaJson();
        if (empty($json)) {
            return array();
        }
        return json_decode($json, true);
    }

    /**
     * @return Collection|Classroom[]
     */
    public function getClassrooms(): Collection
    {
        return $this->classrooms;
    }

    public function addClassroom(Classroom $classroom): self
    {
        if (!$this->classrooms->contains($classroom)) {
            $this->classrooms[] = $classroom;
            $classroom->setOwner($this);
        }

        return $this;
    }

    public function removeClassroom(Classroom $classroom): self
    {
        if ($this->classrooms->contains($classroom)) {
            $this->classrooms->removeElement($classroom);
            // set the owning side to null (unless already changed)
            if ($classroom->getOwner() === $this) {
                $classroom->setOwner(null);
            }
        }

        return $this;
    }

}
