<?php
namespace MainBundle\EntityManager;

use MainBundle\Entity\Client;
use MainBundle\Entity\User;
use MainBundle\Entity\ScoreMode;
use MainBundle\Entity\Abstracts\AbstractUserScore;
use MainBundle\Data\OrderType;
use MainBundle\Data\ScoreSubmitType;
use MainBundle\Data\SystemScoreKey;

abstract class AbstractUserScoreManager extends AbstractEntityManager
{
	
	/**
	 * @return AbstractUserScore
	 */
	abstract protected function createUserScore(): AbstractUserScore;
	
	/**
	 * 
	 * @param \DateTime $time
	 * @return \DateTime
	 */
	abstract public function getPeriodStartByTime(\DateTime $time): \DateTime;
	
	/**
	 * @return \Doctrine\Common\Persistence\ObjectRepository
	 */
	abstract protected function getEntityRepo(): \Doctrine\ORM\EntityRepository;

	public $maxScoreModeCountPerClient = 100;

	public $_debugNow = null;

	public function getModeRepo()
	{
		return $this->getRepo('MainBundle:ScoreMode');
	}
	
	public function getScoreMode(Client $client, $scoreKey, $createIfNotExist): ?ScoreMode
	{
		$query = $this->getModeRepo()->createQueryBuilder('m')
		->where('IDENTITY(m.client) = :cid')
		->andwhere('m.scoreKey = :key')
		->setParameter('cid', $client->getId())
		->setParameter('key', $scoreKey)
		;
		$scoreMode = $query->getQuery()->getOneOrNullResult();
		if(empty($scoreMode) && $createIfNotExist) {

			$this->designContract($this->listScoreModesCount($client, true) < $this->maxScoreModeCountPerClient, 'Number of score modes of the project exceeds limit of ' . $this->maxScoreModeCountPerClient);

			$scoreMode = new ScoreMode();
			$scoreMode->setClient($client)
			->setName($scoreKey)
			->setScoreKey($scoreKey)
			->setCreateTime(new \DateTime())
			;
			$this->getDocMgr()->persist($scoreMode);
			$this->getDocMgr()->flush();
		}
		return $scoreMode;
	}

	public function listScoreModesCount(Client $client)
	{
		$query = $this->getModeRepo()->createQueryBuilder('m')
		->select('count(m)')
		->where('IDENTITY(m.client) = :cid')
		->andwhere('m.deleteTime is null')
		->andWhere('m.scoreKey not in (:systemKeys)')
		->setParameter('cid', $client->getId())
		->setParameter('systemKeys', SystemScoreKey::listAll())
		;
		return intval($query->getQuery()->getSingleScalarResult());
	}

	/**
	 * @return \MainBundle\Entity\ScoreMode[]
	 */
	public function listScoreModesByKeys(Client $client, $keys)
	{
		$query = $this->getModeRepo()->createQueryBuilder('m')
		->where('IDENTITY(m.client) = :cid')
		->andwhere('m.deleteTime is null')
		->andWhere('m.scoreKey in (:keys)')
		->setParameter('cid', $client->getId())
		->setParameter('keys', $keys)
		;
		return $query->getQuery()->getResult();
	}

	/**
	 * @return \MainBundle\Entity\ScoreMode[]
	 */
	public function listScoreModes(Client $client)
	{
		$query = $this->getModeRepo()->createQueryBuilder('m')
		->where('IDENTITY(m.client) = :cid')
		->andwhere('m.deleteTime is null')
		->andWhere('m.scoreKey not in (:systemKeys)')
		->setParameter('cid', $client->getId())
		->setParameter('systemKeys', SystemScoreKey::listAll())
		;
		return $query->getQuery()->getResult();
	}

	public function listPublicScoreModes(Client $client)
	{
		$query = $this->getModeRepo()->createQueryBuilder('m')
		->where('IDENTITY(m.client) = :cid')
		->andwhere('m.showLength != 0')
		->andwhere('m.disabled = 0') // disabled = 0 we assume deleteTime is null as well
		->setParameter('cid', $client->getId())
		;
		return $query->getQuery()->getResult();
	}

	public function setScoreModeDefaultTime(Client $client, $scoreKey)
	{
		$scoreMode = $this->getScoreMode($client, $scoreKey, false);
		$this->designContract(!empty($scoreMode), 'The score key does not exist.');
		$scoreMode->setDefaultTime(new \DateTime());
		$this->getDocMgr()->flush();
		return $scoreMode;
	}

	public function updateScoreMode(Client $client, $scoreKey, $data)
	{
		$this->designContract(!empty($scoreKey), 'Score Key can not be empty.');
		$scoreMode = $this->getScoreMode($client, $scoreKey, true);

		if(!empty($scoreMode->getDeleteTime())) {
			$this->designContract($this->listScoreModesCount($client, true) < $this->maxScoreModeCountPerClient, 'Number of score modes of the project exceeds limit of ' . $this->maxScoreModeCountPerClient);
			$scoreMode->setDeleteTime(null)->setDisabled(false);
		}

		if(!empty($data['name'])) {
			$scoreMode->setName($data['name']);
		}
		if(isset($data['order'])) {
			switch($data['order']) {
				case OrderType::HIGH_TO_LOW:
				case OrderType::LOW_TO_HIGH:
					$scoreMode->setOrderType($data['order']);
					break;
			}
		}
		if(isset($data['length'])) {
			$scoreMode->setShowLength(max(0, min(30, $data['length'])));
		}
		if(isset($data['disabled'])) {
			$scoreMode->setDisabled($data['disabled']);
		}
		if(isset($data['toDollarRate'])) {
			$rate = floor($data['toDollarRate']);
			if($rate < 0) {
				$rate = 0;
			}
			$scoreMode->setToDollarRate($rate);
		}
		$this->getDocMgr()->flush();
		return $scoreMode;
	}

	public function deleteScoreMode(Client $client, $scoreKey)
	{
		$scoreMode = $this->getScoreMode($client, $scoreKey, false);
		$this->designContract(!empty($scoreMode), 'The score key does not exist.');
		$this->designContract(empty($scoreMode->getDeleteTime()), 'The score key is deleted already.');
		$scoreMode->setDeleteTime(new \DateTime())->setDisabled(true);
		$this->getDocMgr()->flush();
		return $scoreMode;
	}

	public function convertScoreModesToJson($scoreModes)
	{
		$list = array();
		foreach($scoreModes as $scoreMode) {
			$list[] = $scoreMode->exportJson();
		}
		return $list;
	}


	/**
	 * 
	 * @param User $user
	 * @param ScoreMode $scoreMode
	 * @param interger $score
	 * @param integer $submitType
	 * @param boolean $flush
	 * @return {userScore: AbstractUserScore, change: integer}
	 */
	public function submitScore(User $user, ScoreMode $scoreMode, $score, $submitType, $flush)
	{
		$now = new \DateTime();
		if(!empty($this->_debugNow)) {
			$now = $this->_debugNow;
		}
		$userScore = $this->getUserScore($user, $scoreMode, $now);
		$oldScore = 0;
		
		if(!empty($userScore))
		{
			$oldScore = $userScore->getScore();
			switch($submitType)
			{
				case ScoreSubmitType::SUBMIT_KEEP_HIGHEST:
					if($score > $userScore->getScore()) {
						$userScore->setScore($score)->setTime($now);
					}
					break;
				case ScoreSubmitType::SUBMIT_KEEP_LOWEST:
					if($score < $userScore->getScore()) {
						$userScore->setScore($score)->setTime($now);
					}
					break;
				case ScoreSubmitType::SUBMIT_ADD:
					if($score != 0) {
						$userScore->setScore($userScore->getScore() + $score)->setTime($now);
					}
					break;
				case ScoreSubmitType::SUBMIT_ADD_SINCE:
					if($score != 0) {
						$userScore->setScore($userScore->getScore() + $score);
					}
					break;
				case ScoreSubmitType::SUBMIT_OVERWRITE:
					if($score != $userScore->getScore()) {
						$userScore->setScore($score)->setTime($now);
					}
					break;
			}
		}
		else
		{
			$userScore = $this->createUserScore();
			$userScore->setUser($user)
			->setScoreMode($scoreMode)
			->setPeriodStart($this->getPeriodStartByTime($now))
			->setScore($score)
			->setTime($now)
			;
			
			$this->getDocMgr()->persist($userScore);
		}
		
		if($flush) {
			$this->getDocMgr()->flush();
		}
		
		return array(
			'userScore' => $userScore,
			'change' => $userScore->getScore() - $oldScore
		);
	}
	
	/**
	 *
	 * @param User $user
	 * @param ScoreMode $scoreMode
	 * @param interger $score
	 * @param integer $submitType
	 * @param \DateTime $timeInPeriod
	 * @param boollean $flush
	 * @return AbstractUserScore
	 */
	public function syncScore(User $user, ScoreMode $scoreMode, $score, $submitType, $timeInPeriod, $flush)
	{
		$userScore = $this->getUserScore($user, $scoreMode, $timeInPeriod);
		$oldScore = 0;
		
		if($userScore)
		{
			$oldScore = $userScore->getScore();
			switch($submitType)
			{
				case ScoreSubmitType::SUBMIT_ADD:
				case ScoreSubmitType::SUBMIT_KEEP_HIGHEST:
					if($score > $userScore->getScore())
					{
						$userScore->setScore($score)
						->setTime($timeInPeriod);
					}
					break;
				case ScoreSubmitType::SUBMIT_KEEP_LOWEST:
					if($score < $userScore->getScore())
					{
						$userScore->setScore($score)
						->setTime($timeInPeriod);
					}
					break;
				case ScoreSubmitType::SUBMIT_ADD_SINCE:
					if($score > $userScore->getScore())
					{
						$userScore->setScore($score);
					}
					break;
				case ScoreSubmitType::SUBMIT_OVERWRITE:
					if($timeInPeriod->getTimestamp() > $userScore->getTime()->getTimestamp())
					{
						$userScore->setScore($score)
						->setTime($timeInPeriod);
					}
					break;
			}
		}
		else
		{
			$userScore = $this->createUserScore();
			$userScore->setUser($user)
			->setScoreMode($scoreMode)
			->setPeriodStart($this->getPeriodStartByTime($timeInPeriod))
			->setScore($score)
			->setTime($timeInPeriod)
			;
				
			$this->getDocMgr()->persist($userScore);
		}

		if($flush) {
			$this->getDocMgr()->flush();
		}
		
		return $userScore;
	}
	
	/**
	 * 
	 * @param User $user
	 * @param ScoreMode $scoreMode
	 * @param \DateTime $timeInPeriod
	 * @return AbstractUserScore
	 */
	public function getUserScore(User $user, ScoreMode $scoreMode, ?\DateTime $timeInPeriod)
	{
		if(empty($timeInPeriod)) {
			$timeInPeriod = new \DateTime();
		}
			
		$query = $this->getEntityRepo()->createQueryBuilder('s')
		->where('IDENTITY(s.user) = :uid')
		->andwhere('IDENTITY(s.scoreMode) = :mid')
		->andwhere('s.periodStart = :period')
		->setParameter('uid', $user->getId())
		->setParameter('mid', $scoreMode->getId())
		->setParameter('period', $this->getPeriodStartByTime($timeInPeriod))
		;
		
		return $query->getQuery()->getOneOrNullResult();
	}

	/**
	 * @return \MainBundle\Entity\Abstracts\AbstractUserScore[]
	 */
	public function listCurrencyScoresByUser(Client $client, User $user)
	{
		$query = $this->getEntityRepo()->createQueryBuilder('s')
		->select('s,m')
		->join('s.scoreMode', 'm')
		->where('IDENTITY(m.client) = :cid')
		->andwhere('m.deleteTime is null')
		->andWhere('s.user = :user')
		->andWhere('m.toDollarRate > 0')
		->setParameter('cid', $client->getId())
		->setParameter('user', $user)
		;
		return $query->getQuery()->getResult();
	}

	/**
	 * @return \MainBundle\Entity\ScoreMode[]
	 */
	public function listCurrencyScoreModes(Client $client)
	{
		$query = $this->getModeRepo()->createQueryBuilder('m')
		->where('IDENTITY(m.client) = :cid')
		->andwhere('m.deleteTime is null')
		->andWhere('m.toDollarRate > 0')
		->setParameter('cid', $client->getId())
		;
		return $query->getQuery()->getResult();
	}
	
	/**
	 * 
	 * @param ScoreMode $scoreMode
	 * @param integer $orderType
	 * @param integer $start
	 * @param integer $length
	 * @param \DateTime $timeInPeriod
	 */
	public function listScores(ScoreMode $scoreMode, $orderType, $start, $length, $timeInPeriod)
	{
		/* @var $builder QueryBuilder */
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
		->join('s.user', 'u')
		->where('IDENTITY(s.scoreMode) = :mid')
		->setParameter('mid', $scoreMode->getId())
		->setFirstResult($start)
		->setMaxResults($length)
		;

		if(!empty($timeInPeriod)) {
			
			$builder->andwhere('s.periodStart = :period')
				->setParameter('period', $this->getPeriodStartByTime($timeInPeriod))
			;
		}
		
		if($orderType == OrderType::HIGH_TO_LOW) {
			$builder->orderBy('s.score', 'DESC');
		} else if($orderType == OrderType::LOW_TO_HIGH) {
			$builder->orderBy('s.score', 'ASC');
		} else if($orderType == OrderType::NEW_TO_OLD) {
			$builder->orderBy('s.time', 'DESC');
		} else if($orderType == OrderType::OLD_TO_NEW) {
			$builder->orderBy('s.time', 'ASC');
		}

		return $builder->getQuery()->getResult();
	}

	public function listScoresAndRankAroundUser(ScoreMode $scoreMode, User $user, $orderType, \DateTime $time, $shiftStart, $length)
	{
		$userScore = empty($scoreMode) ? null : $this->getUserScore($user, $scoreMode, $time);
		if(empty($userScore)) {
			$myScore = false;
			$rank = 1;
		} else {
			$myScore = $userScore->getScore();
			$rank = $this->getScoreRank($scoreMode, $myScore, $orderType, $time);
		}
		
		$scores = $this->listScores($scoreMode, $orderType, max(0, $rank - 1 + $shiftStart), $length, $time);
		
		$rankStart = $rank;
		$list = array();
		/** @var AbstractUserScore $score */
		foreach($scores as $score) {
			$list[] = $score->exportJson();
						
			if($myScore !== false) {
				if($orderType == OrderType::HIGH_TO_LOW) {
					if($score->getScore() > $myScore) {
						--$rankStart;
					}
				} else {
					if($score->getScore() < $myScore) {
						--$rankStart;
					}
				}	
			}
		}
				
		return $this->generateScoreListResult($list, $rankStart, $this->getPeriodStartByTime($time));
	}
	/**
	 * 
	 * @param \MainBundle\Entity\ScoreMode[] $scoreModes
	 * @param \MainBundle\Entity\User $user
	 * @param \DateTime $timeInPeriod
	 * @return \MainBundle\Entity\Abstracts\AbstractUserScore[]
	 */
	public function listScoresByScoreModes($scoreModes, User $user, $timeInPeriod = null)
	{
		if(is_null($timeInPeriod))
			$timeInPeriod = new \DateTime();

		/* @var $builder QueryBuilder */
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
		->where('s.user = :user')
		->andWhere('s.scoreMode in (:modes)')
		->andwhere('s.periodStart = :period')
		->setParameter('user', $user)
		->setParameter('modes', $scoreModes)
		->setParameter('period', $this->getPeriodStartByTime($timeInPeriod))
		;

		return $builder->getQuery()->getResult();
	}

	/**
	 * 
	 * @param ScoreMode $scoreMode
	 * @param string[] $usernames
	 * @param \DateTime $timeInPeriod
	 */
	public function listScoresByUsernames(ScoreMode $scoreMode, $usernames, $timeInPeriod)
	{
		if(is_null($timeInPeriod))
			$timeInPeriod = new \DateTime();
		
		/* @var $builder QueryBuilder */
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
		->join('s.user', 'u')
		->where('IDENTITY(s.scoreMode) = :mid')
		->andwhere('s.periodStart = :period')
		->andwhere('u.username in (:usernames)')
		->setParameter('mid', $scoreMode->getId())
		->setParameter('usernames', $usernames)
		->setParameter('period', $this->getPeriodStartByTime($timeInPeriod))
		;

		return $builder->getQuery()->getResult();
	}

	/**
	 * 
	 * @param ScoreMode $scoreMode
	 * @param integer $orderType
	 * @param integer $start
	 * @param integer $length
	 * @param \DateTime $timeInPeriod
	 */
	public function getScoresTotal(ScoreMode $scoreMode, $timeInPeriod)
	{
		if(is_null($timeInPeriod))
			$timeInPeriod = new \DateTime();
		
		/* @var $builder QueryBuilder */
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
		->select('sum(s.score)')
		->where('IDENTITY(s.scoreMode) = :mid')
		->andwhere('s.periodStart = :period')
		->setParameter('mid', $scoreMode->getId())
		->setParameter('period', $this->getPeriodStartByTime($timeInPeriod))
		;

		$result = $builder->getQuery()->getSingleScalarResult();
		return floatval($result);
	}
	
	/**
	 *
	 * @param ScoreMode $scoreMode
	 * @param integer $score
	 * @param integer $orderType
	 * @param \DateTime $timeInPeriod
	 */
	public function getScoreRank(ScoreMode $scoreMode, $score, $orderType, $timeInPeriod)
	{
		if(is_null($timeInPeriod))
			$timeInPeriod = new \DateTime();
		
		/* @var $builder QueryBuilder */
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
		->select('count(s)')
		->where('IDENTITY(s.scoreMode) = :mid')
		->andwhere('s.periodStart = :period')
		->setParameter('mid', $scoreMode->getId())
		->setParameter('period', $this->getPeriodStartByTime($timeInPeriod))
		;
		
		if($orderType == OrderType::HIGH_TO_LOW)
		{
			$builder->andWhere('s.score > :score')
			->setParameter('score', $score);
		}
		else if($orderType == OrderType::LOW_TO_HIGH)
		{
			$builder->andWhere('s.score < :score')
			->setParameter('score', $score);
		}
		
		return intval($builder->getQuery()->getSingleScalarResult()) + 1;
	}

    public function listScoresHistory(User $user, ScoreMode $scoreMode, \DateTime $startTime, \DateTime $endTime) {
        $startPeriod = $this->getPeriodStartByTime($startTime);
        $endPeriod = $this->getPeriodStartByTime($endTime);
        $builder = $this->getEntityRepo()->createQueryBuilder('s')
			->where('IDENTITY(s.scoreMode) = :mid')
			->andWhere('s.user = :user')
			->andWhere('s.periodStart >= :start')
			->andWhere('s.periodStart <= :end')
			->setParameter('user', $user)
			->setParameter('mid', $scoreMode->getId())
			->setParameter('start', $startPeriod)
			->setParameter('end', $endPeriod)
            ->setMaxResults(366)
		;
		return $builder->getQuery()->getResult();
	}
	
	public function listScoresHistoryByUsernames($usernames, ScoreMode $scoreMode, \DateTime $startTime, \DateTime $endTime) {
        $startPeriod = $this->getPeriodStartByTime($startTime);
        $endPeriod = $this->getPeriodStartByTime($endTime);
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
			->join('s.user', 'u')
			->where('IDENTITY(s.scoreMode) = :mid')
			->andWhere('u.username in (:usernames)')
			->andWhere('s.periodStart >= :start')
			->andWhere('s.periodStart <= :end')
			->setParameter('usernames', $usernames)
			->setParameter('mid', $scoreMode->getId())
			->setParameter('start', $startPeriod)
			->setParameter('end', $endPeriod)
            ->setMaxResults(366 * count($usernames))
		;
		return $builder->getQuery()->getResult();
    }
	
	/**
	 * 
	 * @param Client $client
	 * @return \Doctrine\ORM\mixed
	 */
	public function cleanByClient(Client $client)
	{
		/* @var $builder QueryBuilder */
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
		->delete()
		->join('s.scoreMode', 'm')
		->where('IDENTITY(m.client) = :cid')
		->setParameter('cid', $client->getId())
		;
		return $builder->getQuery()->execute();
	}
	
	/**
	 * 
	 * @param Client $client
	 * @param User $user
	 * @param string $scoreKey
	 */
	public function cleanByUser(Client $client, User $user, $scoreKey)
	{
		/* @var $builder QueryBuilder */
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
		->delete()
		->join('s.scoreMode', 'm')
		->where('IDENTITY(m.client) = :cid')
		->andWhere('IDENTITY(s.user) = :uid')
		->setParameter('cid', $client->getId())
		->setParameter('uid', $user->getId())
		;
		if(!empty($scoreKey))
		{
			$builder->andWhere('m.scoreKey = :scoreKey')
			->setParameter('scoreKey', $scoreKey);
		}
		
		return $builder->getQuery()->execute();
	}
	/**
	 *
	 * @param Client $client
	 * @param User $user
	 * @param string $scoreKey
	 */
	public function cleanByScoreKey(Client $client, $scoreKey)
	{
		/* @var $builder QueryBuilder */
		$builder = $this->getEntityRepo()->createQueryBuilder('s')
		->delete()
		->join('s.scoreMode', 'm')
		->where('IDENTITY(m.client) = :cid')
		->andWhere('m.scoreKey = :scoreKey')
		->setParameter('scoreKey', $scoreKey)
		->setParameter('cid', $client->getId())
		;
	
		return $builder->getQuery()->execute();
	}

	public function convertScoresToJson($scores)
	{
		$list = array();
		/** @var \MainBundle\Entity\Abstracts\AbstractUserScore $score */
		foreach($scores as $score) {
			$list[] = $score->exportJson();
		}
		return $list;
	}
	public function convertScoresToJsonShort($scores)
	{
		$list = array();
		/** @var \MainBundle\Entity\Abstracts\AbstractUserScore $score */
		foreach($scores as $score) {
			$list[] = $score->exportJsonShort();
		}
		return $list;
	}

	public function generateScoreListResult($scores, $rank, ?\DateTime $periodStart)
	{
		return array(
				'scores' => $scores,
				'rankStart' => $rank,
				'periodStart' => empty($periodStart) ? 0 : $periodStart->getTimestamp()
		);
	}
}