<?php

namespace MainBundle\EntityManager;

use Google_Service_Classroom_CourseWork;
use MainBundle\Entity\Classroom;
use MainBundle\Entity\Classwork;
use MainBundle\Entity\User;
use MainBundle\Entity\UserLicense;

class ClassroomManager extends AbstractEntityManager
{
    private $client;

    private $service;

    private $userManager;

    private $user;

    private $licenseManager;

    private $clientManager;

    public function getThisRepo(): \Doctrine\ORM\EntityRepository
    {
        return $this->getRepo('MainBundle:Classroom');
    }

    public function getWorkRepo(): \Doctrine\ORM\EntityRepository
    {
        return $this->getRepo('MainBundle:Classwork');
    }

    public function initUserToken($userToken)
    {
        return $this->initUser($this->getUserManager()->validateUserToken($userToken));
    }

    public function initUser(User $user)
    {
        $this->user = $user;
    }

    public function getUser(): User
    {
        return $this->user;
    }

    public function getLicenseManager(): UserLicenseManager
    {
        if(empty($this->licenseManager)) {
            $this->licenseManager = new UserLicenseManager($this->controller);
        }
        return $this->licenseManager;
    }

    public function getUserManager(): UserManager
    {
        if(empty($this->userManager)) {
            $this->userManager = new UserManager($this->controller);
        }
        return $this->userManager;
    }

    public function getClientManager(): ClientManager
    {
        if(empty($this->clientManager)) {
            $this->clientManager = new ClientManager($this->controller);
        }
        return $this->clientManager;
    }

    public function getClient(): \Google_Client
    {
        if (empty($this->client)) {
            $client = new \Google_Client();
            $client->setApplicationName('Google Classroom API PHP Quickstart');
            $client->setScopes([
                \Google_Service_Classroom::CLASSROOM_COURSES_READONLY,
                \Google_Service_Classroom::CLASSROOM_ROSTERS_READONLY,
                \Google_Service_Classroom::CLASSROOM_COURSEWORK_ME,
                \Google_Service_Classroom::CLASSROOM_COURSEWORK_STUDENTS,
            ]);
            $client->setAuthConfig([
                'client_id' => $this->controller->getYmlParameter('google_app_id'),
                'client_secret' => $this->controller->getYmlParameter('google_app_secret'),
            ]);
            $client->setAccessType('online');
            $this->client = $client;
        }
        return $this->client;
    }

    public function getService(): \Google_Service_Classroom
    {
        if(empty($this->service)) {
            $this->prepareGoogleClient();
            $this->service = new \Google_Service_Classroom($this->getClient());
        }
        return $this->service;
    }

    public function getWpUserId()
    {
        $wpUser = $this->getUserManager()->getWpUserByUser($this->user);
        return $wpUser->getId();
    }

    private function prepareGoogleClient() {
        $client = $this->getClient();
        $rawToken = $this->getUserManager()->getOauthRawToken($this->user);
        $this->designContract(!empty($rawToken), 'RELOGIN');
        $client->setAccessToken($this->getUserManager()->getOauthRawToken($this->user));

        if ($client->isAccessTokenExpired()) {
            if($client->getRefreshToken()) {
                $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
                $this->getUserManager()->saveOauthRawToken($this->user, $client->getAccessToken());
            } else {
                $this->designContract(false, 'RELOGIN');
            }
        }
    }

    public function serviceListCourses()
    {
        $response = $this->getService()->courses->listCourses();
        return $response->getCourses();
    }

    public function serviceGetCourse($courseId)
    {
        $course = $this->getService()->courses->get($courseId);
        return $course;
    }

    public function serviceListStudents($courseId)
    {
        $response = $this->getService()->courses_students->listCoursesStudents($courseId);
        return $response->getStudents();
    }
    
    public function serviceCreateClasswork($courseId, $contentCode, $workInfo)
    {
        $game = $this->getClientManager()->getByCode($contentCode);
        $work = new Google_Service_Classroom_CourseWork();
        $work->setTitle($workInfo['title']);
        $work->setDescription($workInfo['body']);
        $work->setWorkType('ASSIGNMENT');
        $work->setState('PUBLISHED');
        $response = $this->getService()->courses_courseWork->create($courseId, $work);
        return $response;
    }
    public function serviceTurninClasswork(Classwork $classwork, $message): \Google_Service_Classroom_ClassroomEmpty
    {
        $submissions = $this->getService()->courses_courseWork_studentSubmissions->listCoursesCourseWorkStudentSubmissions($classwork->getClassroom()->getCode(), $classwork->getCode());
        $this->designContract(!empty($submissions), 'No submittions found');
        $submission = $submissions->getStudentSubmissions()[0];
        $body = new \Google_Service_Classroom_ModifyAttachmentsRequest([
            "link" => [
                "url" => "/classroom/student_work/". $classwork->getId() . "/" . $this->user->getUsername(),
                "title" => $message,
            ]
        ]);

        $this->getService()->courses_courseWork_studentSubmissions->modifyAttachments(
            $classwork->getClassroom()->getCode(),
            $classwork->getCode(),
            $submission->getId(),
            $body
        );

        $body = new \Google_Service_Classroom_TurnInStudentSubmissionRequest();
        $response = $this->getService()->courses_courseWork_studentSubmissions->turnIn($classwork->getClassroom()->getCode(), $classwork->getCode(), $submission->getId(), $body);
        return $response;
    }
    
    // database access ------------------------
    public function listClassroomByOwner(User $user)
    {
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('c')
            ->where('c.owner = :owner')
            ->setParameter('owner', $user)
        ;
        return $builder->getQuery()->getResult();
    }

    public function getClassroom($id):? Classroom
    {
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('c')
            ->where('c.id = :id')
            ->setParameter('id', $id)
        ;
        return $builder->getQuery()->getOneOrNullResult();
    }

    public function createClassroom(User $owner, $service, $code, $name, $originJson): Classroom
    {
        $classroom = new Classroom();
        $classroom->setName($name)
            ->setCode($code)
            ->setService($service)
            ->setOwner($owner)
            ->setCreateTime(new \DateTime())
            ->setOriginJson(json_encode($originJson))
            ;
        $this->getDocMgr()->persist($classroom);
        $this->getDocMgr()->flush();
        return $classroom;
    }

    public function listClassroomStudents($classroomId)
    {
        $classroom = $this->getClassroom($classroomId);
        $this->designContract($classroom->getOwner()->getId() == $this->user->getId(), 'Not your classroom');
        $students = $this->serviceListStudents($classroom->getCode());
        $googleIds = array_column($students, 'userId');
        return $this->getUserManager()->listByGoogleIds($googleIds);
    }

    public function getClasswork($classroomId, $contentCode)
    {
        $builder = $this->getWorkRepo()->createQueryBuilder('w')
        ->select('w,r,c')
        ->join('w.classroom', 'r')
        ->join('w.client', 'c')
        ->where('r.id = :rid')
        ->andwhere('c.code = :clientCode')
        ->setParameter('rid', $classroomId)
        ->setParameter('clientCode', $contentCode)
        ;
        $result = $builder->getQuery()->getResult();
        return empty($result) ? null : $result[0];
    }

    private function getClassworkInfoByUserLicense(UserLicense $userLicense)
    {
        $this->getWpService()->loadWordpress();
        $postSlug = \get_post_meta($userLicense->getLicense(), 'ClassworkSlug');
        $postSlug = $postSlug[0];
        $gameUrl = \get_post_meta($userLicense->getLicense(), 'Contentlocation');
        $gameUrl = $gameUrl[0];
        $posts = \get_posts(['name' => $postSlug, 'numberposts' => 1]);
        if(!empty($posts)) {
            $post = $posts[0];
            return [
                'title' => $post->post_title,
                'body' => $post->post_content,
            ];
        }
        return [
            'title' => $userLicense->getLicense(),
            'body' => "<a href=\"$gameUrl\">Play the game</a> and submit your score.",
        ];
    }

    public function createClasswork($classroomId, $contentCode)
    {
        $client = $this->getClientManager()->getByCode($contentCode);
        $this->designContract(!empty($client), 'contentCode does not exist:' . $contentCode);
        $wpUserId = $this->getWpUserId();
        $userLicense = $this->getLicenseManager()->getUserLicenseByContentCode($wpUserId, $contentCode);
        $this->designContract(!empty($userLicense), 'Not licensed');
        $classroom = $this->getClassroom($classroomId);
        $this->designContract($classroom->getOwner()->getId() == $this->user->getId(), 'Not your classroom');
        
        $workInfo = $this->getClassworkInfoByUserLicense($userLicense);

        $work = $this->serviceCreateClasswork($classroom->getCode(), $contentCode, $workInfo);
        
        $classwork = new Classwork();
        $classwork->setClassroom($classroom)
            ->setCreateTime(new \DateTime(intval($work->getCreationTime())))
            ->setCode($work->getId())
            ->setClient($client)
            ;
        $this->getDocMgr()->persist($classwork);
        $this->flush();
        return $classwork;
    }

    public function listClassworkByClassroomId($classroomId)
    {
        $builder = $this->getWorkRepo()->createQueryBuilder('w')
            ->select('w,r,c')
            ->join('w.classroom', 'r')
            ->join('w.client', 'c')
            ->where('r.id = :rid')
            ->setParameter('rid', $classroomId)
            ;
        return $builder->getQuery()->getResult();

    }

    public function listClassroomsByCodes($roomCodes)
    {
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('c')
        ->where('c.code in (:codes)')
        ->setParameter('codes', $roomCodes);
        return $builder->getQuery()->getResult();
    }

    public function listStudentClassrooms()
    {
        $rooms = $this->serviceListCourses();
        $roomCodes = [];
        /** @var \Google_Service_Classroom_Course $room */
        foreach($rooms as $room) {
            $roomCodes[] = $room->getId();
        }
        return $this->listClassroomsByCodes($roomCodes);
    }

    public function listClassworkByStudent()
    {
        $classrooms = $this->listStudentClassrooms();
        $classworkList = [];
        /** @var Classroom $classroom */
        foreach($classrooms as $classroom) {
            $thisList = $this->listClassworkByClassroomId($classroom->getId());
            $classworkList = array_merge($classworkList, $thisList);
        }
        return $classworkList;
    }

    public function turnInClasswork(Classwork $classwork, $message): \Google_Service_Classroom_ClassroomEmpty
    {
        return $this->serviceTurninClasswork($classwork, $message);
    }

    public function getStudentClasswork($contentCode):? Classwork
    {
        $classrooms = $this->listStudentClassrooms();
        /** @var Classroom $classroom */
        foreach($classrooms as $classroom) {
            $works = $this->listClassworkByClassroomId($classroom->getId());
            /** @var Classwork $classwork */
            foreach($works as $classwork) {
                if($classwork->getClient()->getCode() == $contentCode) {
                    return $classwork;
                }
            }
        }
        return null;
    }
    
}
