<?php

namespace MainBundle\EntityManager;

use MainBundle\Data\SysConst;
use MainBundle\Entity\Client;
use WP_Query;

class ClientManager extends AbstractEntityManager
{
    /**
     * (non-PHPdoc)
     * @see \MainBundle\EntityManager\AbstractEntityManager::getThisRepo()
     */
    public function getThisRepo(): \Doctrine\ORM\EntityRepository
    {
        return $this->getRepo('MainBundle:Client');
    }

    public function getByCode($code): ?Client
    {
        return $this->getThisRepo()->findOneBycode($code);
    }

    public function getOrCreateByCode($code): Client
    {
        $client = $this->getByCode($code);

        if (empty($client)) {
            $client = new Client();
            $client->setCode($code)
                ->setName($code);

            $this->getDocMgr()->persist($client);
            $this->getDocMgr()->flush();
        }
        return $client;
    }

    public function getSystemClientByClient($clientCode, $ownername): Client
    {
        $code = SysConst::CLIENT_CODE . '/client/' . $clientCode;
        return $this->createClient($code, $ownername);
    }

    public function getSystemClientByOwner($ownername): Client
    {
        $code = SysConst::CLIENT_CODE . '/owner/' . $ownername;
        return $this->createClient($code, $ownername);
    }

    public function listByCodes($codes)
    {
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('c')
            ->where('c.code in (:codes)')
            ->setParameter('codes', $codes);
        return $builder->getQuery()->getResult();
    }

    public function getAppPost($appCode): ?\WP_POST
    {
        $query = new WP_Query([
            'post_type' => 'content',
            'meta_query' => [
                [
                    'key' => 'ContentCode',
                    'value' => $appCode
                ]
            ]
        ]);
        if ($query->have_posts()) {
            return $query->the_post();
        }
        return null;
    }

    public function getAppVersionByCode($appCode)
    {
        $query = new WP_Query([
            'post_type' => 'content',
            'meta_query' => [
                [
                    'key' => 'ContentCode',
                    'value' => $appCode
                ]
            ]
        ]);
        /** @var \WP_POST $post */
        global $post;

        if ($query->have_posts()) {
            $query->the_post();
            return $this->getAppVersionByPost($post);
        }
        return '';
    }

    public function getAppVersionByPost(\WP_POST $post)
    {
        $version = get_post_meta($post->ID, 'ContentVersion', true);
        if (empty($version)) {
            $version = '';
        }
        return $version;
    }

    public function getAppLicenseWithPost($appCode)
    {
        $query = new WP_Query([
            'post_type' => 'content',
            'meta_query' => [
                [
                    'key' => 'ContentCode',
                    'value' => $appCode
                ]
            ]
        ]);
        $userRole = 'loading';
        /** @var \WP_POST $licensedPost */
        $licensedPost = null;
        /** @var \MainBundle\Entity\UserLicense $user_license */
        $user_license = null;

        /** @var \WP_POST $post */
        global $post;

        if ($query->have_posts()) {
            $userRole = 'guest';
            if (is_user_logged_in()) {
                $query->the_post();
                $classroomManager = new ClassroomManager($this->controller);
                $wpUser =  wp_get_current_user();
                $user = $classroomManager->getUserManager()->convertWpUserToUser($wpUser);
                $classroomManager->initUser($user);
                if ($classroomManager->getUserManager()->isUserTeacher($user)) {
                    $userRole = 'teacher';
                    $user_license = $classroomManager->getLicenseManager()->getUserLicenseToAccessPostId($wpUser->ID, $post->ID);
                    if (!empty($user_license)) {
                        $licensedPost = $post;
                    }
                } else if ($classroomManager->getUserManager()->isUserStudent($user)) {
                    $userRole = 'student';
                    $classwork = $classroomManager->getStudentClasswork($appCode);
                    if (!empty($classwork)) {
                        $licensedPost = $post;
                    }
                } else {
                    $userRole = 'user';
                    $user_license = $classroomManager->getLicenseManager()->getUserLicenseToAccessPostId($wpUser->ID, $post->ID);
                    if (!empty($user_license)) {
                        $licensedPost = $post;
                    }
                }
            }
        }

        if (empty($licensedPost)) {
            return null;
        } else {
            return [
                'license' => $user_license,
                'post' => $licensedPost
            ];
        }
    }
}
