<?php

namespace MainBundle\EntityManager;

use MainBundle\Entity\UserLicense;

class UserLicenseManager extends AbstractEntityManager
{
    private $userManager;

    public function getUserManager(): UserManager
    {
        if (empty($this->userManager)) {
            $this->userManager = new UserManager($this->controller);
        }
        return $this->userManager;
    }

    public function getThisRepo(): \Doctrine\ORM\EntityRepository
    {
        return $this->getRepo('MainBundle:UserLicense');
    }

    public function getUserLicenseToAccessPostId($wpUserId, $postId): ?UserLicense
    {
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('ul')
            ->where('ul.wpUserId = :uid')
            ->andwhere('ul.license = :license')
            ->andwhere('ul.maxaccess > 0')
            ->setParameter('uid', $wpUserId)
            ->setParameter('license', $postId)
            ->setMaxResults(1);
        $now = time();
        $result = $builder->getQuery()->getResult();
        /** @var UserLicense $userLicense */
        $userLicense = null;
        /** @var UserLicense $userData */
        foreach ($result as $userData) {
            $endDate = intval($userData->getEndDate());
            if ($now < $endDate) {
                $userLicense = $userData;
                break;
            }
        }
        return $userLicense;
    }

    public function decreaseUserLicenseMaxaccess(?UserLicense $userLicense)
    {
        if (!empty($userLicense)) {
            $maxaccess = max(0, $userLicense->getMaxaccess() - 1);
            $userLicense->setMaxaccess($maxaccess);
            $this->flush();
        }
    }

    public function convertContentCodeToPostId($contentCode)
    {
        $this->getWpService()->loadWordpress();
        $query = new \WP_Query(['meta_key' => 'ContentCode', 'meta_value' => $contentCode, 'post_type' => 'content']);
        $post_id = 0;
        if ($query->have_posts()) {
            $query->the_post();
            $post_id = $query->post->ID;
        }
        \wp_reset_postdata();
        return $post_id;
    }

    public function getUserLicenseContentCode(UserLicense $userLicense): ?string
    {
        if (empty($userLicense)) {
            return null;
        }
        $this->getWpService()->loadWordpress();
        $postId = $userLicense->getLicense();
        $contentCode = \get_post_meta($postId, 'ContentCode', true);
        return $contentCode;
    }

    public function getUserLicenseByContentCode($wpUserId, $contentCode): ?UserLicense
    {
        $postId = $this->convertContentCodeToPostId($contentCode);
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('ul')
            ->where('ul.wpUserId = :uid')
            ->andWhere('ul.license = :postId')
            ->setParameter('uid', $wpUserId)
            ->setParameter('postId', $postId);
        $now = time();
        $result = $builder->getQuery()->getResult();
        /** @var UserLicense $userLicense */
        $userLicense = null;
        /** @var UserLicense $userLicense */
        foreach ($result as $userData) {
            $endDate = intval($userData->getEndDate());
            if ($now < $endDate) {
                $userLicense = $userData;
                break;
            }
        }
        return $userLicense;
    }

    public function listUserLicenses($wpUserId)
    {
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('ul')
            ->where('ul.wpUserId = :uid')
            ->setParameter('uid', $wpUserId);
        $now = time();
        $result = $builder->getQuery()->getResult();
        /** @var UserLicense $userLicense */
        $list = [];
        /** @var UserLicense $userData */
        foreach ($result as $userData) {
            $endDate = intval($userData->getEndDate());
            if ($now < $endDate) {
                $list[] = $userData;
                break;
            }
        }
        return $list;
    }

    public function listLicensedContentCodes($wpUserId)
    {
        $userLicenses = $this->listUserLicenses($wpUserId);
        $codes = [];
        foreach ($userLicenses as $userLicense) {
            $code = $this->getUserLicenseContentCode($userLicense);
            if (!empty($code)) {
                $codes[] = $code;
            }
        }
        return $codes;
    }
}
