<?php

namespace MainBundle\EntityManager;

use MainBundle\Data\KidsUsername;
use MainBundle\Entity\User;
use MainBundle\Libs\NameGenerator\NameGenerator;
use MainBundle\Utils\StringUtil;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

class UserManager extends AbstractEntityManager
{
    public function getThisRepo(): \Doctrine\ORM\EntityRepository
    {
        return $this->getRepo('MainBundle:User');
    }

    public function convertWpUserToUser($wpUser): User
    {
        $wpUserData = $wpUser->data;
        $user = $this->createUser($wpUserData->user_login, $wpUserData->user_email);
        if (empty($user->getId()) || $user->getService() == 'wordpress') {
            $user->setPassword($wpUserData->user_pass)
                ->setNickname($wpUserData->user_nicename)
                ->setProfileUrl($wpUserData->user_url)
                ->setService('wordpress');
            $this->getDocMgr()->persist($user);
            $this->flush();
        }
        return $user;
    }

    public function getWpUserByUser(User $user)
    {
        return $this->getWpService()->userManager->findOneBy(['login' => $user->getUsername()]);
    }

    public function createUser($username, $email): User
    {
        $user = $this->getByUsername($username);
        if (empty($user)) {
            $fosManager = $this->controller->getObject('fos_user.user_manager');
            /* @var $user User */
            $user = $fosManager->createUser();
            $user->setUsername($username)
                ->setUsernameCanonical(strtolower($username))
                ->setEmail($email)
                ->setEmailCanonical(strtolower($email));
        }
        return $user;
    }

    public function getKidsUser($username): ?User
    {
        $username = KidsUsername::generateUsername($username);
        return $this->getByUsername($username);
    }
    public function createKidsUser($username, $passcode, $email): User
    {
        $username = KidsUsername::generateUsername($username);
        $user = $this->getKidsUser($username);
        $this->designContract(empty($user), 'The username has been taken');
        $this->designContract(!empty($username), 'Not a valid username');

        $user = $this->createUser($username, KidsUsername::generateEmail($username));
        $user->setPassword($passcode)
            ->setNickname(KidsUsername::generateNickname($username))
            ->setProfileUrl($this->generateUrl('user_index', array('username' => $username), UrlGeneratorInterface::ABSOLUTE_URL))
            ->setService('app');
        $this->getDocMgr()->persist($user);
        $this->flush();

        $this->saveUserToWp($user, empty($email) ? '' : $email);

        return $user;
    }
    public function validateKidsUser($username, $passcode): User
    {
        $user = $this->getKidsUser($username);
        $this->designContract(!empty($user) && $user->getService() == 'app', 'The user does not exist');
        $this->designContract($passcode == $user->getPassword(), 'The passcode does not match');
        return $user;
    }

    public function getAnonymousUser(): User
    {
        $username = 'anonymous@guest';
        $user = $this->getByUsername($username);
        if (empty($user)) {
            $user = $this->createUser($username, 'anonymous@springbaystudio.net');
            $user->setPassword(StringUtil::generateRandomString(10))
                ->setNickname('Anonymous')
                ->setProfileUrl($this->generateUrl('app_index', array(), UrlGeneratorInterface::ABSOLUTE_URL))
                ->setService('app');
            $this->getDocMgr()->persist($user);
            $this->flush();
        }
        $this->saveUserToWp($user);
        return $user;
    }

    public function getAnonymousRandomUser(): User
    {
        $nickname = NameGenerator::getName();
        $username = $nickname . '@guest';
        $user = $this->getByUsername($username);
        if (empty($user)) {
            $user = $this->createUser($username, str_replace([' '], ['_'], $nickname) . '@springbaystudio.net');
            $user->setPassword(StringUtil::generateRandomString(10))
                ->setNickname($nickname)
                ->setProfileUrl($this->generateUrl('app_index', array(), UrlGeneratorInterface::ABSOLUTE_URL))
                ->setService('app');
            $this->getDocMgr()->persist($user);
            $this->flush();
        }
        $this->saveUserToWp($user);
        return $user;
    }

    private function saveUserToWp(User $user, $email = null)
    {
        $wpService = $this->getWpService();
        $wpUserManager = $wpService->userManager;
        $wpUser = $wpUserManager->findOneBy(['login' => $user->getUsername()]);
        if (empty($wpUser)) {
            if ($email === null) {
                $email = $user->getEmail();
            }
            $wpUser = $wpUserManager->create();
            $wpUser->setLogin($user->getUsername())
                ->setNicename($user->getNickname())
                ->setEmail($email)
                ->setPass($user->getPassword())
                ->setRegistered(new \DateTime())
                ->setDisplayName($user->getNickname())
                ->setUrl($user->getProfileUrl())
                ->setActivationKey('')
                ->setStatus(0);
            $wpUserManager->save($wpUser, true);
        }
    }

    /**
     *
     * @param string $email
     * @return User
     */
    public function getByEmail($email)
    {
        return $this->getThisRepo()->findOneByemail_canonical(strtolower($email));
    }
    /**
     *
     * @param string $username
     * @return User
     */
    public function getByUsername($username)
    {
        return $this->getThisRepo()->findOneByusername($username);
    }

    /**
     *
     * @param string $username
     * @return User
     */
    public function listByUsernames($usernames)
    {
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('u')
            ->where('u.username in (:usernames)')
            ->setParameter('usernames', $usernames);
        return $builder->getQuery()->getResult();
    }

    public function listByGoogleIds($googleIds)
    {
        /* @var $search QueryBuilder */
        $builder = $this->getThisRepo()->createQueryBuilder('u')
            ->where('u.google_id in (:ids)')
            ->setParameter('ids', $googleIds);
        return $builder->getQuery()->getResult();
    }

    public function updateProfileIcon(User $user, $iconFile)
    {
        $user->setIconFile($iconFile);
        $this->getDocMgr()->flush();
    }

    public function search($term, $start, $length)
    {
        $users = array();
        if (!empty($term)) {

            /* @var $search QueryBuilder */
            $search = $this->getThisRepo()->createQueryBuilder('u')
                ->where('u.nickname like :likeTerm')
                ->orWhere('u.email = :term')
                ->setParameter('likeTerm', $term . '%')
                ->setParameter('term', $term)
                ->setFirstResult($start)
                ->setMaxResults($length);
            $users = array_merge($users, $search->getQuery()->getResult());
        }
        return $users;
    }

    public function setUserBlocked(User $user, $block)
    {
        $user->setBlocked($block);
        $this->getDocMgr()->flush();
    }

    public function findUserBy($criteria)
    {
        $search = $this->getThisRepo()->createQueryBuilder('u');
        foreach ($criteria as $key => $value) {
            $search->andwhere("u.$key = :value")->setParameter('value', $value);
        }
        return $search->getQuery()->getOneOrNullResult();
    }

    public function addUserSettings(User $user, $settings)
    {
        if (!$user->hasSetting($settings)) {
            $user->setSettings($settings | $user->getSettings());
            $this->getDocMgr()->flush();
            return true;
        }
        return false;
    }

    public function removeUserSettings(User $user, $settings)
    {
        $settings = 0xFFFFFFFF - $settings;
        $user->setSettings($settings & $user->getSettings());
        $this->getDocMgr()->flush();
    }

    public function validateUserToken($userToken): User
    {
        try {
            $params = explode('|', $userToken);
            $username = $params[0];
            $user = $this->getByUsername($username);
        } catch (\Exception $e) {
            $this->designContract(false, 'Invalid userToken');
        }
        $this->designContract(!empty($user) && $user->getAccessToken() == $userToken, 'Invalid userToken');
        return $user;
    }

    public function getOauthRawToken(User $user)
    {
        $directory = __DIR__ . '/../../../var/login/';
        $logFile = $directory . $user->getUsername() . '_oauth_response.log';
        if (file_exists($logFile)) {
            $content = file_get_contents($logFile);
            $start = strpos($content, ':');
            $content = substr($content, $start + 1);
            $json = json_decode($content, true);
            return $json['rawToken'];
        }
        return null;
    }

    public function saveOauthRawToken(User $user, $token)
    {
        $directory = __DIR__ . '/../../../var/login/';
        $logFile = $directory . $user->getUsername() . '_oauth_response.log';
        if (file_exists($logFile)) {
            $oldContent = file_get_contents($logFile);
            $start = strpos($oldContent, ':') + 1;
            $content = substr($oldContent, $start + 1);
            $json = json_decode($content, true);
            $json['rawToken'] = $token;
            file_put_contents($logFile, substr($oldContent, 0, $start) . json_encode($content));
        }
        return null;
    }

    public function isUserTeacher(?User $user): bool
    {
        if (!empty($user)) {
            $directory = __DIR__ . '/../../../var/login/';
            $logFile = $directory . $user->getUsername() . '_oauth_response.log';
            if (file_exists($logFile)) {
                $content = file_get_contents($logFile);
                return strpos($content, 'https://www.googleapis.com/auth/classroom.rosters.readonly') !== false;
            }
        }
        return false;
    }
    public function isWpUserTeacher($wpUser): bool
    {
        return $this->isUserTeacher($this->convertWpUserToUser($wpUser));
    }
    public function isUserStudent(?User $user): bool
    {
        if (!empty($user)) {
            $directory = __DIR__ . '/../../../var/login/';
            $logFile = $directory . $user->getUsername() . '_oauth_response.log';
            if (file_exists($logFile)) {
                $content = file_get_contents($logFile);
                return  strpos($content, 'https://www.googleapis.com/auth/classroom.rosters.readonly') === false &&
                        strpos($content, 'https://www.googleapis.com/auth/classroom.coursework.me') !== false;
            }
        }
        return false;
    }
    public function isWpUserStudent($wpUser): bool
    {
        return $this->isUserStudent($this->convertWpUserToUser($wpUser));
    }
}
