<?php

namespace MainBundle\Libs\S3;

use MainBundle\Controller\IControllerBase;
use MainBundle\Libs\Common\BaseManager;
use MainBundle\Utils\StringUtil;
use Symfony\Component\Finder\Iterator\RecursiveDirectoryIterator;

class S3Syncer extends BaseManager
{
    public $bucket;

    public $cacheDirectory;

    private $filesToDeleteArchive;

    /**
     *
     * @var MainBundle\EntityManager\S3FileToDeleteManager
     */
    private $s3FileToDeleteManager;

    public function __construct(IControllerBase $controller)
    {
        parent::__construct($controller);
        $s3Gasset = $controller->getYmlParameter('s3.bucket');

        $this->bucket = $controller->getYmlParameter('s3.bucket');
        $this->cacheDirectory = $this->getRootDir() . '/../var/s3/';
        $this->filesToDeleteArchive = $this->cacheDirectory . $s3Gasset . '/filesToDelete.txt';
    }

    /**
     * @return MainBundle\EntityManager\S3FileToDeleteManager
     */
    public function getS3FileToDeleteManager()
    {
        if (empty($this->s3FileToDeleteManager)) {
            $this->s3FileToDeleteManager = new \MainBundle\EntityManager\S3FileToDeleteManager($this->controller);
        }
        return $this->s3FileToDeleteManager;
    }

    public function processFilesToDeleteOnAws()
    {
        $filesToDelete = $this->getS3FileToDeleteManager()->listFilesToDelete(1000);
        $objects = array();
        foreach ($filesToDelete as $file) {
            $objects[] = array('Key' => $file->getFilename());
            $this->getS3FileToDeleteManager()->setFileDeleted($file);
        }
        if(count($objects)) {
            $json = array(
                'Bucket' => $this->bucket,
                'Delete' => array('Objects' => $objects),
            );
            $result = $this->controller->getService('aws.s3')->deleteObjects($json);

            $this->getS3FileToDeleteManager()->flush();

            return $result;
        }
        return false;
    }

    public function addFileToDeleteOnAws($filename, $flush)
    {
        $this->getS3FileToDeleteManager()->addFileToDeleteOnAws($filename, $flush);
    }

    public function syncFolderToAws($dirname, $dry = false)
    {
        $s3Keys = array();
        if (is_dir($dirname)) {
            $it = new RecursiveDirectoryIterator($dirname, RecursiveDirectoryIterator::SKIP_DOTS);
            $files = new \RecursiveIteratorIterator($it, \RecursiveIteratorIterator::CHILD_FIRST);
            foreach ($files as $file) {
                if ($file->isDir()) {
                    //rmdir($file->getRealPath());
                } else {
                    $result = $this->syncFileToAws($file->getRealPath(), $dry);
                    if($result) {
                        $s3Keys[] = $result['key'];
                    }
                }
            }
        }
        return $s3Keys;
    }

    public function syncFileToAws($filename, $dry = false)
    {
        if (is_file($filename)) {
            $base = realpath($this->cacheDirectory);
            $filename = realpath($filename);

            if (strpos($filename, $base) === 0) {
                $key = str_replace('\\', '/', substr($filename, strlen($base)));
                $key = StringUtil::removeStart($key, '/');
                if($dry) {
                    $result = array('dry' => 1);
                } else {
                    $result = $this->controller->getService('aws.s3')->putObject(array(
                        'Bucket' => $this->bucket,
                        'Key' => $key,
                        'SourceFile' => $filename,
                        'ACL' => 'public-read',
                        'StorageClass' => 'STANDARD',
                    ));    
                }
                return array(
                    'key' => $key,
                    'result' => $result
                );
            }
        }
        return false;
    }


    public function syncFileFromAws($s3uri, $dest)
    {
        $dir = dirname($dest);
        if (!is_dir($dir)) {
            mkdir($dir, 0777, true);
        }
        // Save object to a file.
        $result = $this->controller->getService('aws.s3')->getObject(array(
            'Bucket' => $this->bucket,
            'Key' => $s3uri,
            'SaveAs' => $dest,
        ));
        return $result;
    }
}
